<?php
/**
 * Cron Job - بررسی ساعت‌های جفت و ارسال پیام‌های عاشقانه
 * این فایل باید هر دقیقه یکبار اجرا شود
 */

require_once 'config.php';
require_once 'database.php';
require_once 'messages.php';

// دریافت زمان فعلی
$current_hour = date('H');
$current_minute = date('i');
$current_time = date('H:i');

// 🔒 سیستم قفل برای جلوگیری از اجرای مضاعف
$lock_file = __DIR__ . '/cron.lock';
$lock_content = file_exists($lock_file) ? file_get_contents($lock_file) : '';

// اگر قفل برای همین دقیقه وجود داشته باشد، از اجرا مجدد جلوگیری کن
if ($lock_content === $current_time) {
    logActivity("⚠️ Cron already executed for $current_time - skipping");
    die("Already executed for this minute\n");
}

// ثبت زمان فعلی در فایل قفل
file_put_contents($lock_file, $current_time);

logActivity("Cron started at: $current_time");

/**
 * بررسی اینکه آیا ساعت فعلی جفت است یا نه
 */
function isMatchingTime($hour, $minute) {
    // ساعت‌های جفت: هنگامی که ساعت و دقیقه برابر باشند
    // مثل 00:00, 01:01, 02:02, ..., 23:23
    return ($hour == $minute);
}

/**
 * ارسال پیام به یک کاربر
 */
function sendMessageToUser($chat_id, $text) {
    $url = API_URL . "sendMessage";
    
    // اضافه کردن دکمه‌های شیشه‌ای
    $keyboard = [
        'inline_keyboard' => [
            [
                ['text' => '❤️', 'callback_data' => 'like'],
                ['text' => '💝', 'callback_data' => 'love'],
                ['text' => '🔥', 'callback_data' => 'fire']
            ]
        ]
    ];
    
    $data = [
        'chat_id' => $chat_id,
        'text' => $text,
        'parse_mode' => 'HTML',
        'reply_markup' => json_encode($keyboard)
    ];
    
    $options = [
        'http' => [
            'header'  => "Content-type: application/x-www-form-urlencoded\r\n",
            'method'  => 'POST',
            'content' => http_build_query($data),
            'timeout' => 10
        ]
    ];
    
    try {
        $context = stream_context_create($options);
        $result = file_get_contents($url, false, $context);
        
        if ($result) {
            logActivity("Message sent to $chat_id successfully");
            return true;
        } else {
            logError("Failed to send message to $chat_id");
            return false;
        }
    } catch (Exception $e) {
        logError("Exception sending message to $chat_id: " . $e->getMessage());
        return false;
    }
}

// بررسی ساعت جفت
if (isMatchingTime($current_hour, $current_minute)) {
    logActivity("✨ Matching time detected: $current_time");
    
    // دریافت پیام مناسب برای این ساعت
    $message = getMessageForTime($current_time);
    
    if (!$message) {
        logError("❌ No message found for time: $current_time");
        die("No message available\n");
    }
    
    logActivity("📝 Message retrieved for $current_time");
    
    // دریافت تمام کاربران فعال
    $activeUsers = getActiveUsers();
    
    if (!$activeUsers || count($activeUsers) == 0) {
        logActivity("⚠️ No active users found");
        die("No active users\n");
    }
    
    logActivity("👥 Found " . count($activeUsers) . " active users");
    
    // ارسال پیام به همه کاربران فعال
    $successCount = 0;
    $failCount = 0;
    
    foreach ($activeUsers as $user) {
        $chat_id = $user['chat_id'];
        $first_name = $user['first_name'];
        
        logActivity("📤 Attempting to send to: $first_name ($chat_id)");
        
        // شخصی‌سازی پیام با نام کاربر
        $personalizedMessage = str_replace('{name}', $first_name, $message);
        
        if (sendMessageToUser($chat_id, $personalizedMessage)) {
            // افزایش تعداد پیام‌های دریافتی کاربر
            incrementUserMessageCount($chat_id);
            $successCount++;
            logActivity("✅ Sent successfully to $first_name");
            
            // تاخیر کوتاه برای جلوگیری از محدودیت API تلگرام
            usleep(100000); // 100 میلی‌ثانیه
        } else {
            $failCount++;
            logError("❌ Failed to send to $first_name ($chat_id)");
        }
    }
    
    logActivity("📊 Summary: $successCount success, $failCount failed");
    echo "✅ Cron executed: $successCount sent, $failCount failed\n";
    
} else {
    logActivity("⏰ Not a matching time: $current_time (hour=$current_hour, min=$current_minute)");
    echo "⏰ Not matching time: $current_time\n";
}

logActivity("🏁 Cron finished at: " . date('H:i:s'));
echo "Done.\n";
