<?php
/**
 * مدیریت دیتابیس - ذخیره و مدیریت کاربران ربات
 */

require_once 'config.php';

/**
 * اتصال به دیتابیس
 */
function getConnection() {
    // لیست هاست‌های مختلف برای تست
    $hosts = [DB_HOST, '127.0.0.1', 'localhost', 'localhost:/tmp/mysql.sock'];
    
    foreach ($hosts as $host) {
        try {
            $dsn = "mysql:host=" . $host . ";dbname=" . DB_NAME . ";charset=utf8mb4";
            $pdo = new PDO($dsn, DB_USER, DB_PASS);
            $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
            $pdo->setAttribute(PDO::ATTR_DEFAULT_FETCH_MODE, PDO::FETCH_ASSOC);
            return $pdo;
        } catch (PDOException $e) {
            // اگه این هاست کار نکرد، بعدی رو امتحان کن
            continue;
        }
    }
    
    // اگه هیچکدوم کار نکرد، خطا رو لاگ کن
    logError("Database connection failed with all hosts");
    return null;
}

/**
 * ایجاد جدول کاربران (در صورت عدم وجود)
 */
function createTables() {
    $pdo = getConnection();
    if (!$pdo) return false;
    
    try {
        $sql = "CREATE TABLE IF NOT EXISTS users (
            id INT AUTO_INCREMENT PRIMARY KEY,
            chat_id BIGINT NOT NULL UNIQUE,
            first_name VARCHAR(255) NOT NULL,
            active TINYINT(1) DEFAULT 1,
            message_count INT DEFAULT 0,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
        
        $pdo->exec($sql);
        logActivity("Tables created successfully");
        return true;
    } catch (PDOException $e) {
        logError("Failed to create tables: " . $e->getMessage());
        return false;
    }
}

/**
 * اضافه کردن یا به‌روزرسانی کاربر
 */
function addUser($chat_id, $first_name) {
    $pdo = getConnection();
    if (!$pdo) return false;
    
    try {
        // بررسی وجود کاربر
        $stmt = $pdo->prepare("SELECT id FROM users WHERE chat_id = ?");
        $stmt->execute([$chat_id]);
        $user = $stmt->fetch();
        
        if ($user) {
            // به‌روزرسانی کاربر موجود
            $stmt = $pdo->prepare("UPDATE users SET first_name = ?, active = 1, updated_at = NOW() WHERE chat_id = ?");
            $stmt->execute([$first_name, $chat_id]);
            logActivity("User updated: $chat_id");
        } else {
            // اضافه کردن کاربر جدید
            $stmt = $pdo->prepare("INSERT INTO users (chat_id, first_name, active) VALUES (?, ?, 1)");
            $stmt->execute([$chat_id, $first_name]);
            logActivity("New user added: $chat_id - $first_name");
        }
        
        return true;
    } catch (PDOException $e) {
        logError("Failed to add/update user: " . $e->getMessage());
        return false;
    }
}

/**
 * دریافت اطلاعات کاربر
 */
function getUser($chat_id) {
    $pdo = getConnection();
    if (!$pdo) return null;
    
    try {
        $stmt = $pdo->prepare("SELECT * FROM users WHERE chat_id = ?");
        $stmt->execute([$chat_id]);
        return $stmt->fetch();
    } catch (PDOException $e) {
        logError("Failed to get user: " . $e->getMessage());
        return null;
    }
}

/**
 * دریافت تمام کاربران فعال
 */
function getActiveUsers() {
    $pdo = getConnection();
    if (!$pdo) return [];
    
    try {
        $stmt = $pdo->query("SELECT * FROM users WHERE active = 1");
        return $stmt->fetchAll();
    } catch (PDOException $e) {
        logError("Failed to get active users: " . $e->getMessage());
        return [];
    }
}

/**
 * به‌روزرسانی وضعیت کاربر (فعال/غیرفعال)
 */
function updateUserStatus($chat_id, $active) {
    $pdo = getConnection();
    if (!$pdo) return false;
    
    try {
        $stmt = $pdo->prepare("UPDATE users SET active = ?, updated_at = NOW() WHERE chat_id = ?");
        $stmt->execute([$active, $chat_id]);
        logActivity("User status updated: $chat_id - Active: $active");
        return true;
    } catch (PDOException $e) {
        logError("Failed to update user status: " . $e->getMessage());
        return false;
    }
}

/**
 * افزایش تعداد پیام‌های دریافتی کاربر
 */
function incrementUserMessageCount($chat_id) {
    $pdo = getConnection();
    if (!$pdo) return false;
    
    try {
        $stmt = $pdo->prepare("UPDATE users SET message_count = message_count + 1, updated_at = NOW() WHERE chat_id = ?");
        $stmt->execute([$chat_id]);
        return true;
    } catch (PDOException $e) {
        logError("Failed to increment message count: " . $e->getMessage());
        return false;
    }
}

/**
 * دریافت آمار کاربر
 */
function getUserStats($chat_id) {
    $pdo = getConnection();
    if (!$pdo) return null;
    
    try {
        $stmt = $pdo->prepare("SELECT message_count, active, DATE_FORMAT(created_at, '%Y/%m/%d') as join_date FROM users WHERE chat_id = ?");
        $stmt->execute([$chat_id]);
        return $stmt->fetch();
    } catch (PDOException $e) {
        logError("Failed to get user stats: " . $e->getMessage());
        return null;
    }
}

/**
 * دریافت تعداد کل کاربران
 */
function getTotalUsers() {
    $pdo = getConnection();
    if (!$pdo) return 0;
    
    try {
        $stmt = $pdo->query("SELECT COUNT(*) as total FROM users");
        $result = $stmt->fetch();
        return $result['total'];
    } catch (PDOException $e) {
        logError("Failed to get total users: " . $e->getMessage());
        return 0;
    }
}

// نام جایگزین برای سازگاری
function getTotalUsersCount() {
    return getTotalUsers();
}

/**
 * دریافت تعداد کاربران فعال
 */
function getActiveUsersCount() {
    $pdo = getConnection();
    if (!$pdo) return 0;
    
    try {
        $stmt = $pdo->query("SELECT COUNT(*) as total FROM users WHERE active = 1");
        $result = $stmt->fetch();
        return $result['total'];
    } catch (PDOException $e) {
        logError("Failed to get active users count: " . $e->getMessage());
        return 0;
    }
}

/**
 * دریافت لیست تمام کاربران (برای پنل ادمین)
 */
function getAllUsers() {
    $pdo = getConnection();
    if (!$pdo) return [];
    
    try {
        $stmt = $pdo->query("SELECT * FROM users ORDER BY created_at DESC");
        return $stmt->fetchAll();
    } catch (PDOException $e) {
        logError("Failed to get all users: " . $e->getMessage());
        return [];
    }
}

// ایجاد جداول در اولین اجرا
createTables();
